// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "dsp_core.h"
#include "dsp_opdefs.h"
#include "dsp_intops.h"

void DSPInterpreter::__undefined() {
	throw interp_fatal_exception("Unknown opcode!");
}
void DSPInterpreter::_x_undefined() {
	throw interp_fatal_exception("Unknown ext opcode!");
}

START_OPCODE(NOP) {
} END_OPCODE

START_OPCODE(HALT) {
	setHalt();
	m.pause = true;
} END_OPCODE

START_OPCODE(RET) {
	m.npc = m.call.pop();
} END_OPCODE

/*START_OPCODE(RETEQ) {
throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(RETNZ) {
throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE*/

START_OPCODE(RTI) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(CALL) {
	m.call.push(m.npc);
	m.npc = pAddr;
} END_OPCODE

/*START_OPCODE(CALLNE) {
throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE*/

START_OPCODE(JMP) {
	if(cond(pCond)) {
		if(pAddr == m.cpc)
			throw interp_fatal_exception("DSP infinite loop!");
		m.npc = pAddr;
	}
} END_OPCODE

START_OPCODE(DAR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(IAR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(CALLR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(JMPR) {
	m.npc = rReg(pR);
} END_OPCODE

START_OPCODE(SBCLR) {
	if(pI != 2 && pI != 7)
		m.sr &= ~(1 << (pI + 6));
} END_OPCODE

START_OPCODE(SBSET) {
	if(pI != 2 && pI != 7)
		m.sr |= (1 << (pI + 6));
} END_OPCODE

START_OPCODE(S40) {
	m.sr &= ~(1 << 14);
} END_OPCODE

START_OPCODE(S16) {
	m.sr |= (1 << 14);
} END_OPCODE

START_OPCODE(M2) {
	m.sr &= ~(1 << 13);
} END_OPCODE

START_OPCODE(M0) {
	m.sr |= (1 << 13);
} END_OPCODE

START_OPCODE(CLR15) {
	m.sr &= ~(1 << 15);
} END_OPCODE

START_OPCODE(SET15) {
	m.sr |= (1 << 15);
} END_OPCODE

START_OPCODE(LSL) {
	lsl(pAc, pI);
} END_OPCODE

START_OPCODE(LSR) {
	u64 ac = getAc(pAc) & 0xFFFFFFFFFF;
	if(pI != 0) {
		ac >>= (0x40 - pI);
		setAc(pAc, ac);
	}
	setFlags(ac);
} END_OPCODE

START_OPCODE(ASL) {
	lsl(pAc, pI);
} END_OPCODE

START_OPCODE(ASR) {
	s64 ac = getAc(pAc);
	if(pI != 0) {
		ac >>= 0x40 - pI;
		setAc(pAc, ac);
	}
	setFlags(ac);
} END_OPCODE

START_OPCODE(LSL16) {
	lsl(pAc, 16);
} END_OPCODE

START_OPCODE(LSR16) {
	u64 ac = getAc(pAc) & 0xFFFFFFFFFF;
	ac >>= 16;
	setAc(pAc, ac);
	setFlags(ac);
} END_OPCODE

START_OPCODE(ASR16) {
	s64 ac = getAc(pAc);
	ac >>= 16;
	setAc(pAc, ac);
	setFlags(ac);
} END_OPCODE

START_OPCODE(LRI) {
	wReg(pD, pI);
} END_OPCODE

START_OPCODE(LR) {
	wReg(pD, dRead(pM));
} END_OPCODE

START_OPCODE(SR) {
	dWrite(pM, rReg(pS));
} END_OPCODE

START_OPCODE(MRR) {
	WORD data = rReg(pS);
	//copy from wReg
	if(m.sr & SR_M0) {
		switch(pS) {
		/*case 0x10:	//ac0.h
		case 0x11:	//ac1.h
			if((data & 0xFF) != 0)
				if((data & 0xFF) == 0xFF)
					m.gpr[pS + 0xe] |= 0x8000;
				else if(sign8(data))
					m.gpr[pS + 0xe] = 0x8000;
				else
					m.gpr[pS + 0xe] = 0x7FFF;
			else
				m.gpr[pS + 0xe] &= 0x7FFF;
			m.gpr[pD] = data;
			break;*/
		case 0x1e:	//ac0.m
		case 0x1f:	//ac1.m
			if(pD == pS - 0xe) {	//pD is acX.h
				WORD dst = m.gpr[pS - 0xe];
				if(dst != 0 && !sign8(dst)) {
					if(sign16(data) || sign8(data)) {
						if((dst & 0xFF) == 0xFF && (data & 0xFF) == 0xFF) {
						} else if((dst & 0xFF) == 0xFF && sign8(data)) {
							m.gpr[pS] = 0x8000;
						} else if(sign8(dst) && sign16(data)) {
							data = 0;
							m.gpr[pS] = 0x7FFF;
						} else {
							data = 0xFFFF;
							m.gpr[pS] |= 0x8000;
						}
					} else //if((data & 0xFF) == 0) {
						data = 0xFFFF;
				} else if(sign8(dst)) {
					if(sign8(data)) {
						if((dst & 0xFF) != 0xFF)
							data = 0;
					} else if((data & 0xFF) != 0) {
						data = 0;
					}
				} else if(sign16(data)) {
					if(dst == 0)
						data = 0xFFFF;
					else
						data = 0;
				}
			} else if(sign16(data)) {
				data = 0x7FFF;
			} else {
				m.gpr[pS - 0xe] = 0;
			}
			m.gpr[pS - 2] = 0;
			wReg(pD, data);
			break;
		default:
			wReg(pD, data);
		}
	} else {
		wReg(pD, data);
	}
} END_OPCODE

START_OPCODE(SI) {
	dWrite(SEXT8(pM8), pI);
} END_OPCODE

START_OPCODE(LRS) {
	wReg(0x18+pD18, dRead(SEXT8(pM8)));
} END_OPCODE

START_OPCODE(SRS) {
	dWrite(SEXT8(pM8), rReg(0x18+pS18));
} END_OPCODE

START_OPCODE(LRIS) {
	wReg(0x18+pD18, SEXT8(pI));
} END_OPCODE

START_OPCODE(ADDIS) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(CMPIS) {
	s64 ac = getAc(pAc), is = ((s64)(s8)pI) << 16;
	doCmp(ac, is);
} END_OPCODE

START_OPCODE(ANDI) {
	WORD res = m.ACM(pAc) & pI;
	m.ACM(pAc) = res;
	s64 ac = getAc(pAc);
	setFlags(ac, false);
	SETFLAGS(m.sr, SR_EQ, res == 0);
} END_OPCODE

START_OPCODE(ANDF) {
	SETFLAGS(m.sr, SR_LZ, (m.ACM(pAc) & pI) == 0);
} END_OPCODE

START_OPCODE(XORI) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ANDCF) {
	SETFLAGS(m.sr, SR_LZ, (m.ACM(pAc) & pI) == pI);
} END_OPCODE

START_OPCODE(ORI) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ORF) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ADDI) {
	s64 ac = getAc(pAc);
	s64 i = ((s64)((s32)(s16)pI) << 16) & 0xFFFFFF0000;
	bool overflow = (ac & 0xFFFFFF0000) + i > 0xFFFFFF0000;
	ac += i;
	setAc(pAc, ac);
	setFlags(ac);
	SETFLAGS(m.sr, SR_OV, overflow);
	if((ac & 0xFF80000000) == 0x7F80000000)
		m.sr |= 0x82;
} END_OPCODE

START_OPCODE(CMPI) {
	s64 ac = getAc(pAc), i = ((s64)(s16)pI) << 16;
	doCmp(ac, i);
} END_OPCODE

START_OPCODE(ILRR) {
	m.ACM(pAcD) = iRead(m.AR(pArS));
} END_OPCODE

START_OPCODE(ILRRD) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ILRRI) {
	m.ACM(pAcD) = iRead(m.AR(pArS)++);
} END_OPCODE

START_OPCODE(ILRRN) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(LRRI) {
	wReg(pRD, dRead(m.AR(pArS)++));
} END_OPCODE

START_OPCODE(LRRD) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(LRRN) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(LRR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(SRRI) {
	dWrite(m.AR(pArD)++, rReg(pRS));
} END_OPCODE

START_OPCODE(SRRD) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(SRRN) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(SRR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(BLOOP) {
	WORD count = rReg(pR);
	if(count) {
		m.bloopCount++;
		m.call.push(m.npc);
		m.loop_address.push(pAddr);
		m.loop_counter.push(count);
	} else {
		//m.npc = pAddr + 1;	//unsafe; opcodes may be 2 words long.
		m.npc = pAddr + findOpcode(iRead(pAddr)).size;
	}
} END_OPCODE

START_OPCODE(BLOOPI) {
	WORD count = pI;
	if(count) {
		m.bloopCount++;
		m.call.push(m.npc);
		m.loop_address.push(pAddr);
		m.loop_counter.push(count);
	} else {
		m.npc = pAddr + findOpcode(iRead(pAddr)).size;
	}
} END_OPCODE

START_OPCODE(LOOP) {
	WORD counter = rReg(pR);
	WORD loop_address = m.npc;
	if(m.npc >= DSP_IRAMSIZE)
		throw hardware_fatal_exception("DSP loop out of bounds!");
	m.cpc = loop_address;
	WORD opcode = iRead(m.cpc);
	if(g::dsp_log) {  //disassemble
		ostringstream str;
		WORD prev_npc = m.npc;
		disassemble_one(str);
		m.npc = prev_npc;
		DEGUB("\t0x%04X: %04X %s\n", m.cpc, opcode, str.str().c_str());
	}
	while(counter--) {
		m.npc = loop_address;
		(this->*op_main[opcode])();	//execute opcode
	}
} END_OPCODE

START_OPCODE(LOOPI) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(NX) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(DECM) {
	s64 ac = getAc(pAc);
	s64 i = -0x10000;
	bool overflow = (u64)0x10000 <= (u64)ac;
	ac += i;
	setAc(pAc, ac);
	setFlags(ac);
	SETFLAGS(m.sr, SR_OV, overflow);
	if((ac & 0xFF80000000) == 0x7F80000000)
		m.sr |= 0x82;
} END_OPCODE

START_OPCODE(INCM) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(DEC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(INC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(NEG) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(TST) {
	s64 ac = getAc(pAc);
	setFlags(ac);
} END_OPCODE

START_OPCODE(TSTAXH) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(CMP) {
	s64 ac0 = getAc(0), ac1 = getAc(1);
	doCmp(ac0, ac1);
} END_OPCODE

//cmpar in GCEmu
START_OPCODE(CMPAXH) {
	s64 ac = getAc(pAc), axh = (short)rReg(0x1A + pAx);
	axh <<= 16;
	doCmp(ac, axh);
} END_OPCODE

START_OPCODE(CLR) {
	m.gpr[0x10+pAc] = m.gpr[0x1c+pAc] = m.gpr[0x1e+pAc] = 0;
	m.sr = (m.sr & ~0x3F) | 0x24;	//FLAGS
} END_OPCODE

START_OPCODE(CLRP) {
	m.gpr[0x14] = 0x0000;
	m.gpr[0x15] = 0xfff0;
	m.gpr[0x16] = 0x00ff;
	m.gpr[0x17] = 0x0010;
} END_OPCODE

START_OPCODE(MOV) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MOVAX) {
	s32 ax = ((u32)m.AXH(pAxS) << 16) | m.AXL(pAxS);
	s64 ac = ax;
	setAc(pAcD, ac);
	setFlags(ac);
} END_OPCODE

START_OPCODE(MOVR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MOVP) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MOVPZ) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ADDPAXZ) {
	bool useAx = (pMagic == pAxc);
	s32 prod = getProdHM(), source = (useAx) ? (s16)m.AXH(pAxc) : (s16)m.AXH(1 - pAxc);
	s32 result = prod + source;
	bool signAx = sign32(source);
	bool overflow = (signAx && sign8(m.PRODH) && sign24(prod)) ||
		(!sign24(prod) && !sign24(result) && (signAx != sign8(m.PRODH)));
	setAcHM(pAxc, result);
	s64 ac = ((s64)result) << 16;
	setFlags(ac);
	SETFLAGS(m.sr, SR_OV, overflow);
	if(signAx && sign24(prod) && !sign24(result))
		m.sr |= 0x82;
} END_OPCODE

START_OPCODE(ADDP) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(XORR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ORR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ANDR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULX) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULXAC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULXMV) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULXMVZ) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MUL) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULAC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULMV) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULMVZ) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULCAC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULCMV) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MULCMVZ) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ADDR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ADDAX) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(ADD) {
	u64 acD = getAc(pAcD);
	uint pAcD1 = pAcD == 0 ? 1 : 0;
	u64 acD1 = getAc(pAcD1);
	u64 result = acD + acD1;
	bool overflow = (!sign40(acD) && sign40(acD1) && !sign40(result)) ||
		(sign40(acD) && !(!sign40(acD1) && sign40(result)));
	setAc(pAcD, result);
	setFlags(result, true);
	SETFLAGS(m.sr, SR_OV, overflow);
	if((!sign40(acD) && !sign40(acD1) && sign40(result)) ||
		(sign40(acD) && sign40(acD1) && !sign40(result)))
		m.sr |= 0x82;
} END_OPCODE

START_OPCODE(ADDAXL) {
	//copy from ADD and ADDI
	s64 ac = getAc(pAcD);
	u64 i = m.AXL(pAxS);
	u64 result = ac + i;
	bool overflow = (!sign40(ac) && sign40(i) && !sign40(result)) ||
		(sign40(ac) && !(!sign40(i) && sign40(result)));
	setAc(pAcD, result);
	setFlags(result);
	SETFLAGS(m.sr, SR_OV, overflow);
	if((result & 0xFF80000000) == 0x7F80000000)
		m.sr |= 0x82;
} END_OPCODE

START_OPCODE(SUBR) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(SUBAX) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(SUB) {
	s64 acD = getAc(pAcD);
	uint pAcD1 = pAcD == 0 ? 1 : 0;
	s64 acD1 = getAc(pAcD1);
	bool overflow = (u64)acD1 <= (u64)acD;
	s64 result = acD - acD1;
	setAc(pAcD, result);
	setFlags(result, true);
	SETFLAGS(m.sr, SR_OV, overflow);
	if((sign40(acD) && !sign40(acD1) && !sign40(result)) ||
		(!sign40(acD) && sign40(acD1) && sign40(result)))
		m.sr |= 0x82;
} END_OPCODE

START_OPCODE(MADD) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MSUB) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MADDX) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MSUBX) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MADDC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE

START_OPCODE(MSUBC) {
	throw hardware_fatal_exception("DSP unemulated opcode!");
} END_OPCODE


START_EXT_OPCODE(L) {
	WORD address = rReg(pRS);
	wReg(0x18 + pD18, dRead(address));
	wReg(pRS, address + 1);
} END_EXT_OPCODE

START_EXT_OPCODE(LN) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(LS) {
	//$(0x18+D) = MEM[$ar0]
	//$ar0++
	wReg(0x18 + pD18, dRead(m.AR(0)));
	m.AR(0)++;

	//MEM[$ar3] = $acS.m
	//$ar3++
	dWrite(m.AR(3), m.ACM(pAcS));
	m.AR(3)++;
} END_EXT_OPCODE

START_EXT_OPCODE(LSN) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(LSM) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(LSNM) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(SL) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(SLN) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(SLM) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(SLNM) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(S) {
	//MEM[$D] = $(0x1c+S)
	//$S++
	dWrite(m.AR(pRD), m.gpr[0x1c + pS1c]);
	m.AR(pRD)++;
} END_EXT_OPCODE

START_EXT_OPCODE(SN) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(LD) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(LDN) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(LDM) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(LDNM) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(MV) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(DR) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(IR) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(NR) {
	throw hardware_fatal_exception("DSP unemulated ext opcode!");
} END_EXT_OPCODE

START_EXT_OPCODE(NOP) {
} END_EXT_OPCODE


/*#define DEFINE_UNEMULATED_OPCODE(name) void DSPInterpreter::_##name() {\
throw hardware_fatal_exception("DSP unemulated opcode!"); }
DSP_OPCODES(DEFINE_UNEMULATED_OPCODE);
#define DEFINE_UNEMULATED_EXT_OPCODE(name) void DSPInterpreter::_x##name() {\
throw hardware_fatal_exception("DSP unemulated ext opcode!"); }
DSP_EXT_OPCODES(DEFINE_UNEMULATED_EXT_OPCODE);
*/
